<?php
session_start();
include 'conexao.php';

// Redireciona se o usuário não estiver logado.
if (!isset($_SESSION['usuario'])) {
    header("Location: login.php");
    exit;
}

$usuario_id = $_SESSION['usuario']['id'];
$nome_usuario = $_SESSION['usuario']['nome'] ?? 'Usuário';

// --- Lógica de Exclusão de Documento ---
if (isset($_GET['excluir_id'])) {
    $documento_id = intval($_GET['excluir_id']);

    // Primeiro, busque o caminho do arquivo e verifique a posse do documento.
    $stmt_find = $conn->prepare("SELECT caminho_arquivo FROM documentos_enviados WHERE id = ? AND destinatario_id = ?");
    $stmt_find->bind_param("ii", $documento_id, $usuario_id);
    $stmt_find->execute();
    $resultado_find = $stmt_find->get_result();
    
    if ($resultado_find->num_rows > 0) {
        $doc_to_delete = $resultado_find->fetch_assoc();
        $caminho_arquivo = $doc_to_delete['caminho_arquivo'];

        // Se o arquivo existir no servidor, tente excluí-lo.
        if (file_exists($caminho_arquivo)) {
            unlink($caminho_arquivo);
        }

        // Exclua o registro do banco de dados.
        $stmt_delete = $conn->prepare("DELETE FROM documentos_enviados WHERE id = ?");
        $stmt_delete->bind_param("i", $documento_id);
        $stmt_delete->execute();
        $stmt_delete->close();
        
        // Redirecione para a mesma página para atualizar a lista.
        header("Location: documentos_recebidos.php");
        exit;
    }
    $stmt_find->close();
}
// --- Fim da Lógica de Exclusão ---

// Aumenta o tempo de execução do script para lidar com grandes volumes de dados, se necessário.
set_time_limit(300);

// Buscar documentos recebidos.
$stmt = $conn->prepare("
    SELECT
        d.id,
        d.titulo,
        d.caminho_arquivo,
        d.data_envio,
        COALESCE(p.nome, a.nome, u.nome) AS remetente
    FROM
        documentos_enviados d
    LEFT JOIN
        professores p ON d.remetente_id = p.id
    LEFT JOIN
        alunos a ON d.remetente_id = a.id
    LEFT JOIN
        usuarios u ON d.remetente_id = u.id
    WHERE
        d.destinatario_id = ?
    ORDER BY
        d.data_envio DESC
");
$stmt->bind_param("i", $usuario_id);
$stmt->execute();
$resultado = $stmt->get_result();
$documentos = $resultado->fetch_all(MYSQLI_ASSOC);
$stmt->close();
?>

<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Documentos Recebidos</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body {
            font-family: Arial, sans-serif;
            padding: 20px;
            background: #f4f4f4;
            color: #333;
        }
        h2 {
            text-align: center;
            color: #003366;
            margin-bottom: 20px;
        }
        .container {
            max-width: 900px;
            margin: auto;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
            background: white;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
            border-radius: 8px;
            overflow: hidden;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 12px 15px;
            text-align: left;
            vertical-align: middle;
        }
        th {
            background: #003366;
            color: white;
            text-transform: uppercase;
            font-size: 14px;
        }
        tbody tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        tbody tr:hover {
            background-color: #f1f1f1;
        }
        a.download, button.delete-btn {
            color: #007bff;
            text-decoration: none;
            font-weight: bold;
            display: inline-block;
            transition: color 0.3s;
            cursor: pointer;
        }
        a.download:hover {
            color: #0056b3;
            text-decoration: underline;
        }
        button.delete-btn {
            background: none;
            border: none;
            color: #dc3545;
            font-size: 16px;
        }
        button.delete-btn:hover {
            color: #c82333;
        }
        .no-documents {
            text-align: center;
            font-weight: bold;
            margin-top: 30px;
            color: #555;
            padding: 20px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.05);
        }

        /* Estilos para o Modal de Confirmação */
        .modal {
            display: none;
            position: fixed;
            z-index: 1;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.4);
        }
        .modal-content {
            background-color: #fefefe;
            margin: 15% auto;
            padding: 20px;
            border: 1px solid #888;
            width: 80%;
            max-width: 400px;
            text-align: center;
            border-radius: 10px;
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        .modal-buttons {
            margin-top: 20px;
        }
        .modal-buttons button {
            padding: 10px 20px;
            margin: 0 10px;
            border-radius: 5px;
            cursor: pointer;
            font-weight: bold;
        }
        .modal-buttons .confirm {
            background-color: #dc3545;
            color: white;
            border: none;
        }
        .modal-buttons .cancel {
            background-color: #ccc;
            color: #333;
            border: 1px solid #999;
        }

        /* Responsividade da tabela */
        @media screen and (max-width: 768px) {
            table, thead, tbody, th, td, tr {
                display: block;
            }
            thead tr {
                position: absolute;
                top: -9999px;
                left: -9999px;
            }
            tr {
                border: 1px solid #ccc;
                margin-bottom: 10px;
                border-radius: 8px;
            }
            td {
                border: none;
                border-bottom: 1px solid #eee;
                position: relative;
                padding-left: 50%;
                text-align: right;
            }
            td:before {
                position: absolute;
                top: 6px;
                left: 6px;
                width: 45%;
                padding-right: 10px;
                white-space: nowrap;
                text-align: left;
                font-weight: bold;
                color: #555;
            }
            td:nth-of-type(1):before { content: "#"; }
            td:nth-of-type(2):before { content: "Título"; }
            td:nth-of-type(3):before { content: "Remetente"; }
            td:nth-of-type(4):before { content: "Data"; }
            td:nth-of-type(5):before { content: "Arquivo"; }
            td:nth-of-type(6):before { content: "Ação"; }
        }
    </style>
</head>
<body>

<div class="container">
    <h2>📁 Documentos Recebidos - <?= htmlspecialchars($nome_usuario) ?></h2>

    <?php if (count($documentos) > 0): ?>
    <table>
        <thead>
            <tr>
                <th>#</th>
                <th>Título</th>
                <th>Remetente</th>
                <th>Data</th>
                <th>Arquivo</th>
                <th>Acção</th>
            </tr>
        </thead>
        <tbody>
            <?php $i = 1; foreach ($documentos as $doc): ?>
                <tr>
                    <td><?= $i++ ?></td>
                    <td><?= htmlspecialchars($doc['titulo']) ?></td>
                    <td><?= htmlspecialchars($doc['remetente']) ?></td>
                    <td><?= date('d/m/Y H:i', strtotime($doc['data_envio'])) ?></td>
                    <td><a class="download" href="<?= $doc['caminho_arquivo'] ?>" target="_blank">📎 Abrir</a></td>
                    <td><button class="delete-btn" onclick="showModal(<?= $doc['id'] ?>)"><i class="fas fa-trash-alt"></i> Excluir</button></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php else: ?>
        <p class="no-documents">Nenhum documento recebido.</p>
    <?php endif; ?>
</div>

<!-- Modal de Confirmação de Exclusão -->
<div id="deleteModal" class="modal">
    <div class="modal-content">
        <h4>Confirmação de Exclusão</h4>
        <p>Tem certeza de que deseja excluir este documento? Esta ação não pode ser desfeita.</p>
        <div class="modal-buttons">
            <button class="confirm" id="confirmDeleteBtn">Excluir</button>
            <button class="cancel" onclick="hideModal()">Cancelar</button>
        </div>
    </div>
</div>

<script>
    let docIdToDelete = null;

    function showModal(docId) {
        docIdToDelete = docId;
        const modal = document.getElementById('deleteModal');
        modal.style.display = 'block';
    }

    function hideModal() {
        docIdToDelete = null;
        const modal = document.getElementById('deleteModal');
        modal.style.display = 'none';
    }

    document.getElementById('confirmDeleteBtn').addEventListener('click', () => {
        if (docIdToDelete) {
            window.location.href = `documentos_recebidos.php?excluir_id=${docIdToDelete}`;
        }
    });

    // Fechar o modal clicando fora dele
    window.onclick = function(event) {
        const modal = document.getElementById('deleteModal');
        if (event.target == modal) {
            hideModal();
        }
    }
</script>

</body>
</html>